/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include "WLServerInfo.hpp"

#include <errno.h>
#include <string>
#include <cmath>
#include <ilm/ilm_control.h>

using namespace WaylandBackend;

void WLServerInfo::serverinfoCallback(void *data, struct serverinfo *pServerinfo,
                                        uint32_t client_handle)
{
    (void)pServerinfo;
    WLServerInfo* pServerInfo = static_cast<WLServerInfo *>(data);
    pServerInfo->_lmConnect_id = client_handle;
}

const struct serverinfo_listener WLServerInfo::serverinfo_listener =
{
	WLServerInfo::serverinfoCallback
};

WLServerInfo::WLServerInfo(struct wl_event_queue *eventQ)
:_lmServerInfo(NULL)
,_wlEventQ(eventQ)
,_wlID(0)
,_wlInterfaceName("serverinfo")
,_lmConnect_id(0)
,_shellName("LayerManager")
{

}

int
WLServerInfo::init(struct wl_registry* registry, uint32_t name,
                   const char* interface, uint32_t version)
{
    int ret = -EINVAL;
    std::string ifName(interface);

    if (ILM_FAILED == ilm_init())
    {
        /* Mostly wayland server is unavailable*/
        return -EIO;
    }

    if (ifName == _wlInterfaceName)
    {
        _wlID = name;

        ret = 0;

        if (NULL != _wlEventQ)
        {
            struct wl_registry* wrappedRegistry = (struct wl_registry*)
                             wl_proxy_create_wrapper((void *)registry);
            if (NULL != wrappedRegistry)
            {
                wl_proxy_set_queue((struct wl_proxy*)wrappedRegistry, _wlEventQ);
                _lmServerInfo = (struct serverinfo*)wl_registry_bind(registry, name,
                        &serverinfo_interface, version);
                wl_proxy_wrapper_destroy(wrappedRegistry);
            }
        }
        else
        {
            /*
            * It is not good idea to use default queue. default queue can be
            * dispatched by undesired threads
            */
            _lmServerInfo = (struct serverinfo*)wl_registry_bind(registry, name,
                &serverinfo_interface, version);
        }
        if (NULL != _lmServerInfo)
        {
            serverinfo_add_listener(_lmServerInfo, &serverinfo_listener, (void*)this);
            serverinfo_get_connection_id(_lmServerInfo);
        }
        else
        {
            ret = -ENOMEM;
        }
    }

    return ret;
}

uint32_t
WLServerInfo::getWLID()
{
    return _wlID;
}

std::string&
WLServerInfo::getWLName()
{
    return _wlInterfaceName;
}

int
WLServerInfo::getErrno(ilmErrorTypes ilm_error)
{
    int ret;
    switch (ilm_error)
    {
    case ILM_SUCCESS:
        ret = 0;
        break;
    case ILM_FAILED:
        ret = -EINVAL;
        break;
    case ILM_ERROR_INVALID_ARGUMENTS :
        ret = -EINVAL;
        break;
    case ILM_ERROR_ON_CONNECTION :
        ret = -EIO;
        break;
    case ILM_ERROR_RESOURCE_ALREADY_INUSE :
        ret = -EINVAL;
        break;
    case ILM_ERROR_RESOURCE_NOT_FOUND :
        ret = -ENOMEM;
        break;
    case ILM_ERROR_NOT_IMPLEMENTED :
        ret = -ENOTSUP;
        break;
    case ILM_ERROR_UNEXPECTED_MESSAGE :
        ret = -EIO;
        break;
    default:
        ret = -EINVAL;
    }
    return ret;
}

int
WLServerInfo::surfaceCreate(struct wl_surface* wlSurface,
                            unsigned int surfaceId, unsigned int layerId,
                            unsigned int width, unsigned int height)
{
    uint32_t id = (uint32_t) wl_proxy_get_id((struct wl_proxy*)wlSurface);
    uint32_t native_ilm_handle = (_lmConnect_id << 16) | id;
    ilmErrorTypes ilmError;

    ilmError = ilm_surfaceCreate((t_ilm_nativehandle) native_ilm_handle, width,
            height, ILM_PIXELFORMAT_RGBA_8888, &surfaceId);

    if (ILM_SUCCESS == ilmError)
    {
        ilmError = ilm_layerAddSurface((t_ilm_layer)layerId,
                                       (t_ilm_surface)surfaceId);
        if (ILM_SUCCESS == ilmError)
            ilmError = ilm_commitChanges();
    }
    return getErrno(ilmError);
}

int
WLServerInfo::surfaceDestroy(unsigned int surfaceId)
{
    ilmErrorTypes ilmError;

    ilmError = ilm_surfaceRemove(surfaceId);

    return getErrno(ilmError);
}

int
WLServerInfo::surfaceSetVisbility(unsigned int surfaceId, void *shellSurface,
        bool visible)
{
    ilmErrorTypes ilmError;

    (void)shellSurface;
    ilmError = ilm_surfaceSetVisibility((t_ilm_surface)surfaceId, visible);

    if (ILM_SUCCESS == ilmError)
        ilmError = ilm_commitChanges();

    return getErrno(ilmError);
}

void
WLServerInfo::getVisibleLine(int &position, unsigned int &distance)
{
    if(position < 0)
    {
        if(std::abs(position) < distance)
        {
            distance = distance - std::abs(position);
            position = 0;
        }
	else
        {
            //No portion of line is in positive direction
            //Surface is not visible in this case
            distance = 0;
            position = 0;
        }
    }
}

void
WLServerInfo::chooseVisibleLine(int &position, unsigned int &distance)
{
    if(position >= 0)
    {
        //full line is visible i.e. complete surface can
        //be shown on screen as interface does not allow
        // displaying a portion of surface.
        position = 0;
    }
    else
    {
        if(std::abs(position) < distance)
        {
            //choose positive portion of the line.
            position = std::abs(position);
            distance = distance - position;
        }
	else
	{
            //No portion of line is in positive direction.
            //Surface not visible in this case.
            position = 0;
            distance = 0;
        }
    }
}

void
WLServerInfo::getPositiveDestRect(int &destX, int &destY,
                                  unsigned int &destWidth,
                                  unsigned int &destHeight)
{
    getVisibleLine(destX, destWidth);
    getVisibleLine(destY, destHeight);
}

void
WLServerInfo::getPositiveSrcRect(int &sourceX, int &sourceY,
                                 unsigned int &sourceWidth,
                                 unsigned int &sourceHeight)
{
    chooseVisibleLine(sourceX, sourceWidth);
    chooseVisibleLine(sourceY, sourceHeight);
}

int
WLServerInfo::surfaceConfigure(unsigned int surfaceId, void *shellSurface,
                               int x, int y, unsigned int width,
                               unsigned int height)
{
    int sourceX;
    int sourceY;
    int destX;
    int destY;
    unsigned int sourceWidth;
    unsigned int sourceHeight;
    unsigned int destWidth;
    unsigned int destHeight;
    ilmErrorTypes ilmError;

    (void)shellSurface;

    sourceX = x;
    sourceY = y;
    sourceWidth = (unsigned int)width;
    sourceHeight = (unsigned int)height;
    destX = x;
    destY = y;
    destWidth = (unsigned int)width;
    destHeight = (unsigned int)height;

    getPositiveSrcRect(sourceX, sourceY, sourceWidth, sourceHeight);
    getPositiveDestRect(destX, destY, destWidth, destHeight);

    ilmError = ilm_surfaceSetSourceRectangle((t_ilm_surface)surfaceId, sourceX,
                                             sourceY, sourceWidth,
                                             sourceHeight);

    if (ILM_SUCCESS == ilmError)
    {
        ilmError = ilm_surfaceSetDestinationRectangle((t_ilm_surface)surfaceId,
                                             destX, destY, destWidth,
                                             destHeight);
        if (ILM_SUCCESS == ilmError)
            ilmError = ilm_commitChanges();
    }

    return getErrno(ilmError);
}

int
WLServerInfo::surfaceConfigureSynchronous(unsigned int surfaceId,
                                          void *shellSurface, int x, int y,
                                          unsigned int width,
                                          unsigned int height)
{
    int sourceX;
    int sourceY;
    int destX;
    int destY;
    unsigned int sourceWidth;
    unsigned int sourceHeight;
    unsigned int destWidth;
    unsigned int destHeight;
    ilmErrorTypes ilmError;

    (void)shellSurface;

    sourceX = x;
    sourceY = y;
    sourceWidth = (unsigned int)width;
    sourceHeight = (unsigned int)height;
    destX = x;
    destY = y;
    destWidth = (unsigned int)width;
    destHeight = (unsigned int)height;

    getPositiveSrcRect(sourceX, sourceY, sourceWidth, sourceHeight);
    getPositiveDestRect(destX, destY, destWidth, destHeight);

    ilmError = ilm_surfaceSetSourceRectangleDeferred((t_ilm_surface)surfaceId,
                                                     sourceX, sourceY,
                                                     sourceWidth,
                                                     sourceHeight);

    if (ILM_SUCCESS == ilmError)
    {
        ilmError = ilm_surfaceSetDestinationRectangleDeferred(
                                             (t_ilm_surface)surfaceId,
                                             destX, destY, destWidth,
                                             destHeight);
        if (ILM_SUCCESS == ilmError)
            ilmError = ilm_commitChanges();
    }

    return getErrno(ilmError);
}


WLServerInfo::~WLServerInfo()
{
    if (NULL != _lmServerInfo)
        serverinfo_destroy(_lmServerInfo);
}
